// Copyright 2025 SETKA. All Rights Reserved.
#include "ChannelPackerPro.h"
#include "ToolMenus.h"
#include "Widgets/Docking/SDockTab.h"
#include "Widgets/Text/STextBlock.h"
#include "Widgets/Input/SEditableTextBox.h"
#include "Widgets/Input/SButton.h"
#include "Widgets/Input/SNumericEntryBox.h"
#include "Widgets/Layout/SSeparator.h"
#include "Widgets/Layout/SSpacer.h"
#include "Widgets/Layout/SUniformGridPanel.h"
#include "Framework/Docking/TabManager.h"
#include "Framework/Notifications/NotificationManager.h"
#include "Widgets/Notifications/SNotificationList.h"
#include "Styling/AppStyle.h"
#include "Logging/LogMacros.h"
#include "PropertyCustomizationHelpers.h"
#include "Engine/Texture2D.h"
#include "AssetRegistry/AssetRegistryModule.h"
#include "Misc/Paths.h"
#include "ImageUtils.h"
#include "Math/UnrealMathUtility.h"
#include "Math/Float16.h"
#include "Widgets/Input/SComboButton.h"
#include "Widgets/Input/SComboBox.h"
#include "Widgets/Images/SImage.h"
#include "Widgets/Layout/SScrollBox.h"
#include "ContentBrowserModule.h"
#include "IContentBrowserSingleton.h"
#include "ThumbnailRendering/ThumbnailManager.h"
#include "Internationalization/Internationalization.h"
#include "Internationalization/Culture.h"
#include "Misc/ScopedSlowTask.h"

#define LOCTEXT_NAMESPACE "FChannelPackerProModule"

DEFINE_LOG_CATEGORY_STATIC(LogTexturePacker, Log, All);

static const FName ChannelPackerProTabName("ChannelPackerPro");

void FChannelPackerProModule::StartupModule()
{
    // Initialize Compression Options
    CompressionOptions.Add(MakeShared<FString>("Masks"));
    // CompressionOptions.Add(MakeShared<FString>("Grayscale"));
    CompressionOptions.Add(MakeShared<FString>("Default"));
    CurrentCompressionOption = CompressionOptions[1];

    // Initialize Channel Options
    ChannelOptions.Add(MakeShared<FString>("R"));
    ChannelOptions.Add(MakeShared<FString>("G"));
    ChannelOptions.Add(MakeShared<FString>("B"));
    ChannelOptions.Add(MakeShared<FString>("A"));
    ChannelSelectionR = ChannelOptions[0];
    ChannelSelectionG = ChannelOptions[0];
    ChannelSelectionB = ChannelOptions[0];
    ChannelSelectionA = ChannelOptions[0];

    // Register Nomad Tab
    FGlobalTabmanager::Get()->RegisterNomadTabSpawner(ChannelPackerProTabName, FOnSpawnTab::CreateRaw(this, &FChannelPackerProModule::OnSpawnPluginTab))
        .SetDisplayName(LOCTEXT("ChannelPackerProTabTitle", "Texture Channel Packer Pro"))
        .SetMenuType(ETabSpawnerMenuType::Hidden)
        .SetIcon(FSlateIcon(FAppStyle::GetAppStyleSetName(), "Icons.Layout"));

    // This code will execute after your module is loaded into memory; the exact timing is specified in the .uplugin file format
    FToolMenuOwnerScoped OwnerScoped(this);
    UToolMenus* ToolMenus = UToolMenus::Get();

    // Find the 'Tools' menu in the level editor main menu
    UToolMenu* ToolsMenu = ToolMenus->ExtendMenu("LevelEditor.MainMenu.Tools");

    // Add a new section to the 'Tools' menu
    FToolMenuSection& Section = ToolsMenu->AddSection("ChannelPackerPro", LOCTEXT("ChannelPackerProSection", "Texture Packing"));

    // Add a new menu entry to the new section
    Section.AddMenuEntry(
        "PackTextures",
        LOCTEXT("PackTexturesMenuEntry", "Texture Channel Packer Pro"),
        LOCTEXT("PackTexturesMenuEntryTooltip", "Open RGBA Channels packer tool."),
        FSlateIcon(FAppStyle::GetAppStyleSetName(), "Icons.Layout"),
        FUIAction(
            FExecuteAction::CreateLambda([]()
            {
                FGlobalTabmanager::Get()->TryInvokeTab(ChannelPackerProTabName);
            })
        )
    );
}

void FChannelPackerProModule::ShutdownModule()
{
    // This function may be called during shutdown to clean up your module.  For modules that support dynamic reloading,
    // we call this function before unloading the module.
    UToolMenus::UnregisterOwner(this);
    FGlobalTabmanager::Get()->UnregisterNomadTabSpawner(ChannelPackerProTabName);
}

TSharedRef<SDockTab> FChannelPackerProModule::OnSpawnPluginTab(const FSpawnTabArgs& SpawnTabArgs)
{
    TSharedRef<SComboButton> PathPickerComboButton = SNew(SComboButton)
        .ContentPadding(FMargin(2.0f, 2.0f))
        .ButtonContent()
        [
            SNew(SImage)
            .Image(FAppStyle::GetBrush("Icons.FolderClosed"))
        ];

    TWeakPtr<SComboButton> WeakComboButton = PathPickerComboButton;
    PathPickerComboButton->SetOnGetMenuContent(FOnGetContent::CreateLambda([this, WeakComboButton]()
    {
        FContentBrowserModule& ContentBrowserModule = FModuleManager::LoadModuleChecked<FContentBrowserModule>("ContentBrowser");
        FPathPickerConfig PathPickerConfig;
        PathPickerConfig.DefaultPath = OutputPackagePath;
        PathPickerConfig.OnPathSelected = FOnPathSelected::CreateLambda([this, WeakComboButton](const FString& NewPath)
        {
            OutputPackagePath = NewPath;
            if (TSharedPtr<SComboButton> StrongComboButton = WeakComboButton.Pin())
            {
                StrongComboButton->SetIsOpen(false);
            }
        });

        return ContentBrowserModule.Get().CreatePathPicker(PathPickerConfig);
    }));

    const FLinearColor ActiveResColor(0.15f, 0.45f, 1.0f, 1.0f);
    const FLinearColor InactiveResColor = FLinearColor::White;

    return SNew(SDockTab)
        .TabRole(ETabRole::NomadTab)
        [
            SNew(SScrollBox)
            + SScrollBox::Slot()
            [
                SNew(SVerticalBox)

                    // Output Settings Header
                    + SVerticalBox::Slot()
                    .AutoHeight()
                    .Padding(10.0f, 5.0f)
                    [
                        SNew(STextBlock)
                            .Text(LOCTEXT("InputSettingsLabel", "Input"))
                            .Font(FAppStyle::GetFontStyle("PropertyWindow.BoldFont"))
                    ]

            // Channel Inputs in two rows
            + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(10.0f)
            [
                SNew(SVerticalBox)

                // Row 1: R and G
                + SVerticalBox::Slot()
                .AutoHeight()
                .Padding(0.0f, 0.0f, 0.0f, 6.0f)
                [
                    SNew(SHorizontalBox)

                    // Red Column
                    + SHorizontalBox::Slot()
                    .FillWidth(1.0f)
                    .Padding(5.0f, 0.0f)
                    [
                        SNew(SBorder)
                        .BorderImage(FAppStyle::GetBrush("WhiteBrush"))
                        .BorderBackgroundColor(FLinearColor(0.2f, 0.05f, 0.05f, 0.25f)) // subtle dark red
                        .Padding(FMargin(6.0f))
                        [
                            SNew(SVerticalBox)
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                            [
                                SNew(STextBlock)
                                .Text(LOCTEXT("RedChannelLabel", "R Input"))
                                .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            [
                                SNew(SObjectPropertyEntryBox)
                                .AllowedClass(UTexture2D::StaticClass())
                                .ObjectPath_Lambda([this]()
                                {
                                    return InputTextureR.IsValid() ? InputTextureR->GetPathName() : FString();
                                })
                                .OnObjectChanged_Lambda([this](const FAssetData& AssetData)
                                {
                                    InputTextureR = Cast<UTexture2D>(AssetData.GetAsset());
                                })
                                .AllowClear(true)
                                .DisplayThumbnail(true)
                                .ThumbnailPool(UThumbnailManager::Get().GetSharedThumbnailPool())
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 4.0f, 0.0f, 0.0f)
                            [
                                SNew(SUniformGridPanel)
                                .SlotPadding(FMargin(2.0f, 2.0f))
                                + SUniformGridPanel::Slot(0, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("R")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return (ChannelSelectionR == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionR = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(1, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("G")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return (ChannelSelectionR == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionR = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(2, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("B")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return (ChannelSelectionR == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionR = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(3, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("A")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return (ChannelSelectionR == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionR = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                            ]
                        ]
                    ]

                    // Green Column
                    + SHorizontalBox::Slot()
                    .FillWidth(1.0f)
                    .Padding(5.0f, 0.0f)
                    [
                        SNew(SBorder)
                        .BorderImage(FAppStyle::GetBrush("WhiteBrush"))
                        .BorderBackgroundColor(FLinearColor(0.05f, 0.2f, 0.05f, 0.25f)) // subtle dark green
                        .Padding(FMargin(6.0f))
                        [
                            SNew(SVerticalBox)
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                            [
                                SNew(STextBlock)
                                .Text(LOCTEXT("GreenChannelLabel", "G Input"))
                                .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            [
                                SNew(SObjectPropertyEntryBox)
                                .AllowedClass(UTexture2D::StaticClass())
                                .ObjectPath_Lambda([this]()
                                {
                                    return InputTextureG.IsValid() ? InputTextureG->GetPathName() : FString();
                                })
                                .OnObjectChanged_Lambda([this](const FAssetData& AssetData)
                                {
                                    InputTextureG = Cast<UTexture2D>(AssetData.GetAsset());
                                })
                                .AllowClear(true)
                                .DisplayThumbnail(true)
                                .ThumbnailPool(UThumbnailManager::Get().GetSharedThumbnailPool())
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 4.0f, 0.0f, 0.0f)
                            [
                                SNew(SUniformGridPanel)
                                .SlotPadding(FMargin(2.0f, 2.0f))
                                + SUniformGridPanel::Slot(0, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("R")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return (ChannelSelectionG == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionG = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(1, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("G")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return (ChannelSelectionG == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionG = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(2, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("B")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return (ChannelSelectionG == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionG = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(3, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("A")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return (ChannelSelectionG == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionG = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                            ]
                        ]
                    ]
                ]

                // Row 2: B and A
                + SVerticalBox::Slot()
                .AutoHeight()
                .Padding(0.0f, 0.0f, 0.0f, 0.0f)
                [
                    SNew(SHorizontalBox)

                    // Blue Column
                    + SHorizontalBox::Slot()
                    .FillWidth(1.0f)
                    .Padding(5.0f, 0.0f)
                    [
                        SNew(SBorder)
                        .BorderImage(FAppStyle::GetBrush("WhiteBrush"))
                        .BorderBackgroundColor(FLinearColor(0.05f, 0.05f, 0.2f, 0.25f)) // subtle dark blue
                        .Padding(FMargin(6.0f))
                        [
                            SNew(SVerticalBox)
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                            [
                                SNew(STextBlock)
                                .Text(LOCTEXT("BlueChannelLabel", "B Input"))
                                .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            [
                                SNew(SObjectPropertyEntryBox)
                                .AllowedClass(UTexture2D::StaticClass())
                                .ObjectPath_Lambda([this]()
                                {
                                    return InputTextureB.IsValid() ? InputTextureB->GetPathName() : FString();
                                })
                                .OnObjectChanged_Lambda([this](const FAssetData& AssetData)
                                {
                                    InputTextureB = Cast<UTexture2D>(AssetData.GetAsset());
                                })
                                .AllowClear(true)
                                .DisplayThumbnail(true)
                                .ThumbnailPool(UThumbnailManager::Get().GetSharedThumbnailPool())
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 4.0f, 0.0f, 0.0f)
                            [
                                SNew(SUniformGridPanel)
                                .SlotPadding(FMargin(2.0f, 2.0f))
                                + SUniformGridPanel::Slot(0, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("R")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return (ChannelSelectionB == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionB = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(1, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("G")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return (ChannelSelectionB == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionB = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(2, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("B")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return (ChannelSelectionB == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionB = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(3, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("A")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return (ChannelSelectionB == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionB = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                            ]
                        ]
                    ]

                    // Alpha Column
                    + SHorizontalBox::Slot()
                    .FillWidth(1.0f)
                    .Padding(5.0f, 0.0f)
                    [
                        SNew(SBorder)
                        .BorderImage(FAppStyle::GetBrush("WhiteBrush"))
                        .BorderBackgroundColor(FLinearColor(0.12f, 0.12f, 0.12f, 0.25f)) // subtle dark gray for alpha
                        .Padding(FMargin(6.0f))
                        [
                            SNew(SVerticalBox)
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                            [
                                SNew(STextBlock)
                                .Text(LOCTEXT("AlphaChannelLabel", "A Input"))
                                .ToolTipText(LOCTEXT("AlphaChannelTooltip", "Filled with white if empty."))
                                .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            [
                                SNew(SObjectPropertyEntryBox)
                                .AllowedClass(UTexture2D::StaticClass())
                                .ObjectPath_Lambda([this]()
                                {
                                    return InputTextureA.IsValid() ? InputTextureA->GetPathName() : FString();
                                })
                                .OnObjectChanged_Lambda([this](const FAssetData& AssetData)
                                {
                                    InputTextureA = Cast<UTexture2D>(AssetData.GetAsset());
                                })
                                .AllowClear(true)
                                .DisplayThumbnail(true)
                                .ThumbnailPool(UThumbnailManager::Get().GetSharedThumbnailPool())
                            ]
                            + SVerticalBox::Slot()
                            .AutoHeight()
                            .Padding(0.0f, 4.0f, 0.0f, 0.0f)
                            [
                                SNew(SUniformGridPanel)
                                .SlotPadding(FMargin(2.0f, 2.0f))
                                + SUniformGridPanel::Slot(0, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("R")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return (ChannelSelectionA == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionA = ChannelOptions.IsValidIndex(0) ? ChannelOptions[0] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(1, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("G")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return (ChannelSelectionA == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionA = ChannelOptions.IsValidIndex(1) ? ChannelOptions[1] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(2, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("B")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return (ChannelSelectionA == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionA = ChannelOptions.IsValidIndex(2) ? ChannelOptions[2] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                                + SUniformGridPanel::Slot(3, 0)
                                [
                                    SNew(SButton)
                                    .HAlign(HAlign_Center)
                                    .ContentPadding(FMargin(2.0f))
                                    .Text(FText::FromString(TEXT("A")))
                                    .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                                    {
                                        TSharedPtr<FString> Option = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return (ChannelSelectionA == Option) ? ActiveResColor : InactiveResColor;
                                    })
                                    .OnClicked_Lambda([this]()
                                    {
                                        ChannelSelectionA = ChannelOptions.IsValidIndex(3) ? ChannelOptions[3] : nullptr;
                                        return FReply::Handled();
                                    })
                                ]
                            ]
                        ]
                    ]
                ]
            ]

            // Separator
            + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(10.0f, 5.0f)
            [
                SNew(SSeparator)
            ]

            // Output Settings Header
            + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(10.0f, 5.0f)
            [
                SNew(STextBlock)
                .Text(LOCTEXT("OutputSettingsLabel", "Output"))
                .Font(FAppStyle::GetFontStyle("PropertyWindow.BoldFont"))
            ]

                // Resolution
                + SVerticalBox::Slot()
                .AutoHeight()
                .Padding(10.0f, 5.0f)
                [
                    SNew(SVerticalBox)
                    + SVerticalBox::Slot()
                    .AutoHeight()
                    .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                    [
                        SNew(STextBlock)
                        .Text(LOCTEXT("ResolutionLabel", "Resolution"))
                        .ToolTipText(LOCTEXT("ResolutionTooltip", "Valid range: 64 - 8192"))
                        .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                    ]
                    + SVerticalBox::Slot()
                    .AutoHeight()
                    [
                        SNew(SUniformGridPanel)
                        .SlotPadding(FMargin(4.0f, 4.0f))
                        + SUniformGridPanel::Slot(0, 0)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("64")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 64 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 64;
                                return FReply::Handled();
                            })
                        ]
                        + SUniformGridPanel::Slot(1, 0)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("128")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 128 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 128;
                                return FReply::Handled();
                            })
                        ]
                        + SUniformGridPanel::Slot(2, 0)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("256")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 256 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 256;
                                return FReply::Handled();
                            })
                        ]
                        + SUniformGridPanel::Slot(3, 0)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("512")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 512 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 512;
                                return FReply::Handled();
                            })
                        ]
                        + SUniformGridPanel::Slot(0, 1)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("1024")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 1024 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 1024;
                                return FReply::Handled();
                            })
                        ]
                        + SUniformGridPanel::Slot(1, 1)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("2048")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 2048 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 2048;
                                return FReply::Handled();
                            })
                        ]
                        + SUniformGridPanel::Slot(2, 1)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("4096")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 4096 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 4096;
                                return FReply::Handled();
                            })
                        ]
                        + SUniformGridPanel::Slot(3, 1)
                        [
                            SNew(SButton)
                            .HAlign(HAlign_Center)
                            .Text(FText::FromString(TEXT("8192")))
                            .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                            {
                                return TargetResolution == 8192 ? ActiveResColor : InactiveResColor;
                            })
                            .OnClicked_Lambda([this]()
                            {
                                TargetResolution = 8192;
                                return FReply::Handled();
                            })
                        ]
                    ]
                ]

            // Compression Settings
            + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(10.0f, 5.0f)
            [
                SNew(SVerticalBox)
                + SVerticalBox::Slot()
                .AutoHeight()
                .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                [
                    SNew(STextBlock)
                    .Text(LOCTEXT("CompressionLabel", "Compression"))
                    .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                ]
                + SVerticalBox::Slot()
                .AutoHeight()
                [
                    SNew(SUniformGridPanel)
                    .SlotPadding(FMargin(2.0f, 2.0f))
                    + SUniformGridPanel::Slot(0, 0)
                    [
                        SNew(SButton)
                        .HAlign(HAlign_Center)
                        .ToolTipText(LOCTEXT("CompressionTooltipMasks", "Masks: Best for ORM or other packed data. Linear, no sRGB."))
                        .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                        {
                            TSharedPtr<FString> Option = CompressionOptions.IsValidIndex(0) ? CompressionOptions[0] : nullptr;
                            return (CurrentCompressionOption == Option) ? ActiveResColor : InactiveResColor;
                        })
                        .OnClicked_Lambda([this]()
                        {
                            CurrentCompressionOption = CompressionOptions.IsValidIndex(0) ? CompressionOptions[0] : nullptr;
                            return FReply::Handled();
                        })
                        [
                            SNew(STextBlock).Text(FText::FromString(TEXT("Masks")))
                        ]
                    ]
                    + SUniformGridPanel::Slot(1, 0)
                    [
                        SNew(SButton)
                        .HAlign(HAlign_Center)
                        .ToolTipText(LOCTEXT("CompressionTooltipDefault", "Default: Standard compression (sRGB enabled)."))
                        .ButtonColorAndOpacity_Lambda([this, ActiveResColor, InactiveResColor]()
                        {
                            TSharedPtr<FString> Option = CompressionOptions.IsValidIndex(1) ? CompressionOptions[1] : nullptr;
                            return (CurrentCompressionOption == Option) ? ActiveResColor : InactiveResColor;
                        })
                        .OnClicked_Lambda([this]()
                        {
                            CurrentCompressionOption = CompressionOptions.IsValidIndex(1) ? CompressionOptions[1] : nullptr;
                            return FReply::Handled();
                        })
                        [
                            SNew(STextBlock).Text(FText::FromString(TEXT("Default")))
                        ]
                    ]
                ]
            ]

            // Output Path
            + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(10.0f, 5.0f)
            [
                SNew(SVerticalBox)
                + SVerticalBox::Slot()
                .AutoHeight()
                .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                [
                    SNew(STextBlock)
                    .Text(LOCTEXT("OutputPathLabel", "Output Path (e.g. /Game/...)"))
                    .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                ]
                + SVerticalBox::Slot()
                .AutoHeight()
                [
                    SNew(SHorizontalBox)
                    + SHorizontalBox::Slot()
                    .FillWidth(1.0f)
                    .HAlign(HAlign_Fill)
                    [
                        SNew(SEditableTextBox)
                        .Text_Lambda([this] { return FText::FromString(OutputPackagePath); })
                        .OnTextCommitted_Lambda([this](const FText& NewText, ETextCommit::Type) { OutputPackagePath = NewText.ToString(); })
                    ]
                    + SHorizontalBox::Slot()
                    .AutoWidth()
                    [
                        PathPickerComboButton
                    ]
                ]
            ]

            // File Name
            + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(10.0f, 5.0f)
            [
                SNew(SVerticalBox)
                + SVerticalBox::Slot()
                .AutoHeight()
                .Padding(0.0f, 0.0f, 0.0f, 4.0f)
                [
                    SNew(STextBlock)
                    .Text(LOCTEXT("FileNameLabel", "Output Name"))
                    .ToolTipText(LOCTEXT("FileNameTooltip", "For example T_NoiseMasks_Packed."))
                    .Font(FAppStyle::GetFontStyle("PropertyWindow.NormalFont"))
                ]
                + SVerticalBox::Slot()
                .AutoHeight()
                [
                    SNew(SEditableTextBox)
                    .Text_Lambda([this] { return FText::FromString(OutputFileName); })
                    .OnTextCommitted_Lambda([this](const FText& NewText, ETextCommit::Type CommitType)
                    {
                        OutputFileName = NewText.ToString();
                        if (CommitType == ETextCommit::OnEnter || CommitType == ETextCommit::OnUserMovedFocus)
                        {
                            bFileNameManuallyEdited = true;
                        }
                    })
                ]
            ]

            // Spacer
            /* + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(10.0f)
            [
                SNew(SSpacer)
                .Size(FVector2D(0.0f, 10.0f))
            ] */

            // Generate Button
            + SVerticalBox::Slot()
            .AutoHeight()
            .Padding(20.0f)
            .HAlign(HAlign_Fill)
            [
                SNew(SButton)
                .HAlign(HAlign_Center)
                .VAlign(VAlign_Center)
                .ContentPadding(FMargin(0.0f, 10.0f))
                .OnClicked_Lambda([this]()
                {
                    return OnGenerateClicked();
                })
                [
                    SNew(STextBlock)
                    .Text(LOCTEXT("GenerateButtonText", "Pack Channels"))
                    .Font(FAppStyle::GetFontStyle("PropertyWindow.BoldFont"))
                ]
            ]
            ]
        ];
}


FReply FChannelPackerProModule::OnGenerateClicked()
{
    UE_LOG(LogTexturePacker, Log, TEXT("Generating Texture..."));
    UE_LOG(LogTexturePacker, Log, TEXT("Input Red: %s"), InputTextureR.IsValid() ? *InputTextureR->GetPathName() : TEXT("None"));
    UE_LOG(LogTexturePacker, Log, TEXT("Input Green: %s"), InputTextureG.IsValid() ? *InputTextureG->GetPathName() : TEXT("None"));
    UE_LOG(LogTexturePacker, Log, TEXT("Input Blue: %s"), InputTextureB.IsValid() ? *InputTextureB->GetPathName() : TEXT("None"));
    UE_LOG(LogTexturePacker, Log, TEXT("Input Alpha: %s"), InputTextureA.IsValid() ? *InputTextureA->GetPathName() : TEXT("None"));
    UE_LOG(LogTexturePacker, Log, TEXT("Resolution: %d"), TargetResolution);
    UE_LOG(LogTexturePacker, Log, TEXT("Output Path: %s"), *OutputPackagePath);
    UE_LOG(LogTexturePacker, Log, TEXT("File Name: %s"), *OutputFileName);

    // Validation Check 1: At least one input texture
    if (!InputTextureR.IsValid() && !InputTextureG.IsValid() && !InputTextureB.IsValid() && !InputTextureA.IsValid())
    {
        FText Msg = LOCTEXT("ErrorNoTextures", "Please select at least one input texture.");
        ShowNotification(Msg, false);
        return FReply::Handled();
    }

    // Validation Check 2: Output filename is not empty
    if (OutputFileName.IsEmpty())
    {
        FText Msg = LOCTEXT("ErrorNoFileName", "Please specify a file name.");
        ShowNotification(Msg, false);
        return FReply::Handled();
    }

    // Validation Check 3: Resolution is valid
    if (TargetResolution < 1 || TargetResolution > 8192)
    {
        FText Msg = LOCTEXT("ErrorInvalidResolution", "Resolution must be between 1 and 8192.");
        ShowNotification(Msg, false);
        return FReply::Handled();
    }

    FString PackageName = OutputPackagePath;
    if (!PackageName.EndsWith(TEXT("/")))
    {
        PackageName += TEXT("/");
    }
    PackageName += OutputFileName;

    CreateTexture(PackageName, TargetResolution);

    return FReply::Handled();
}

/* void FChannelPackerProModule::AutoGenerateFileName()
{
    if (bFileNameManuallyEdited)
    {
        return;
    }

    TArray<FString> InputNames;
    if (InputTextureR.IsValid()) InputNames.Add(InputTextureR->GetName());
    if (InputTextureG.IsValid()) InputNames.Add(InputTextureG->GetName());
    if (InputTextureB.IsValid()) InputNames.Add(InputTextureB->GetName());
    if (InputTextureA.IsValid()) InputNames.Add(InputTextureA->GetName());

    if (InputNames.Num() == 0)
    {
        return;
    }

    // Find Common Prefix
    FString CommonPrefix = InputNames[0];
    for (int32 i = 1; i < InputNames.Num(); ++i)
    {
        const FString& CurrentName = InputNames[i];
        int32 CommonLen = 0;
        int32 MaxLen = FMath::Min(CommonPrefix.Len(), CurrentName.Len());
        for (int32 CharIdx = 0; CharIdx < MaxLen; ++CharIdx)
        {
            if (CommonPrefix[CharIdx] == CurrentName[CharIdx])
            {
                CommonLen++;
            }
            else
            {
                break;
            }
        }
        CommonPrefix = CommonPrefix.Left(CommonLen);
    }

    FString BaseName;
    if (CommonPrefix.Len() >= 3)
    {
        BaseName = CommonPrefix;
    }
    else
    {
        BaseName = InputNames[0]; // First valid input
    }

    // Enforce "T_" prefix
    if (!BaseName.StartsWith(TEXT("T_")))
    {
        BaseName = TEXT("T_") + BaseName;
    }

    // Remove trailing underscores
    while (BaseName.EndsWith(TEXT("_")))
    {
        BaseName.LeftChopInline(1);
    }

    OutputFileName = BaseName + TEXT("_ORM");
} */

// Helper function to read and resize texture data
static EChannelSelection GetChannelFromSelection(const TSharedPtr<FString>& Selection)
{
    if (Selection.IsValid())
    {
        const FString& Val = *Selection;
        if (Val == "G") return EChannelSelection::G;
        if (Val == "B") return EChannelSelection::B;
        if (Val == "A") return EChannelSelection::A;
    }
    return EChannelSelection::R;
}

static uint8 GetChannelValue(const FColor& Color, EChannelSelection Channel)
{
    switch (Channel)
    {
    case EChannelSelection::G: return Color.G;
    case EChannelSelection::B: return Color.B;
    case EChannelSelection::A: return Color.A;
    case EChannelSelection::R:
    default: return Color.R;
    }
}

// Helper function to read and resize texture data
static TArray<uint8> GetResizedTextureData(UTexture2D* SourceTex, int32 TargetSize, EChannelSelection Channel)
{
    TArray<uint8> ResultData;
    // Lazy Allocation: Do not initialize ResultData yet.

    if (!SourceTex)
    {
        ResultData.Init(0, TargetSize * TargetSize);
        return ResultData;
    }

#if WITH_EDITORONLY_DATA
    // Access Source Data
    int32 SrcWidth = SourceTex->Source.GetSizeX();
    int32 SrcHeight = SourceTex->Source.GetSizeY();
    ETextureSourceFormat SrcFormat = SourceTex->Source.GetFormat();

    // Lock Mip 0
    uint8* SrcData = SourceTex->Source.LockMip(0);
    if (!SrcData)
    {
        UE_LOG(LogTexturePacker, Warning, TEXT("Failed to lock source mip for texture: %s"), *SourceTex->GetName());
        ResultData.Init(0, TargetSize * TargetSize);
        return ResultData;
    }

    int32 NumPixels = SrcWidth * SrcHeight;
    TArray<FColor> SrcColors;
    SrcColors.SetNumUninitialized(NumPixels);

    // Convert input to FColor (BGRA)
    switch (SrcFormat)
    {
    case TSF_BGRA8:
    {
        FMemory::Memcpy(SrcColors.GetData(), SrcData, NumPixels * sizeof(FColor));
        break;
    }
    case TSF_G8:
    {
        const uint8* GrayData = SrcData;
        for (int32 i = 0; i < NumPixels; ++i)
        {
            uint8 Val = GrayData[i];
            SrcColors[i] = FColor(Val, Val, Val, 255);
        }
        break;
    }
    case TSF_G16:
    {
        const uint16* GrayData16 = (const uint16*)SrcData;
        for (int32 i = 0; i < NumPixels; ++i)
        {
            uint8 Val = (uint8)(GrayData16[i] >> 8);
            SrcColors[i] = FColor(Val, Val, Val, 255);
        }
        break;
    }
    case TSF_R16F:
    {
        const FFloat16* Pixel16 = (const FFloat16*)SrcData;
        for (int32 i = 0; i < NumPixels; ++i)
        {
            uint8 Val = (uint8)FMath::Clamp<float>((float)Pixel16[i] * 255.0f, 0.0f, 255.0f);
            SrcColors[i] = FColor(Val, Val, Val, 255);
        }
        break;
    }
    case TSF_R32F:
    {
        const float* Pixel32 = (const float*)SrcData;
        for (int32 i = 0; i < NumPixels; ++i)
        {
            uint8 Val = (uint8)FMath::Clamp<float>(Pixel32[i] * 255.0f, 0.0f, 255.0f);
            SrcColors[i] = FColor(Val, Val, Val, 255);
        }
        break;
    }
    case TSF_RGBA32F:
    {
        const FLinearColor* LinearColors = (const FLinearColor*)SrcData;
        for (int32 i = 0; i < NumPixels; ++i)
        {
            uint8 Val = (uint8)FMath::Clamp<float>(LinearColors[i].R * 255.0f, 0.0f, 255.0f);
            SrcColors[i] = FColor(Val, Val, Val, 255);
        }
        break;
    }
    default:
    {
        UE_LOG(LogTexturePacker, Error, TEXT("Unsupported Source Format: %d for texture: %s"), (int32)SrcFormat, *SourceTex->GetName());
        SourceTex->Source.UnlockMip(0);

        FText ErrorMsg = LOCTEXT("ErrorUnsupportedFormat", "Texture format not supported. Please convert to PNG or TGA.");

        FNotificationInfo Info(ErrorMsg);
        Info.ExpireDuration = 3.0f;
        Info.Image = FAppStyle::GetBrush("Icons.ErrorWithColor");

        TSharedPtr<SNotificationItem> NotificationItem = FSlateNotificationManager::Get().AddNotification(Info);
        if (NotificationItem.IsValid())
        {
            NotificationItem->SetCompletionState(SNotificationItem::CS_Fail);
            NotificationItem->ExpireAndFadeout();
        }

        ResultData.Init(0, TargetSize * TargetSize);
        return ResultData;
    }
    }

    SourceTex->Source.UnlockMip(0);

    // Resize if necessary
    TArray<FColor> ResizedColors;
    if (SrcWidth != TargetSize || SrcHeight != TargetSize)
    {
        ResizedColors.SetNum(TargetSize * TargetSize);
        FImageUtils::ImageResize(SrcWidth, SrcHeight, SrcColors, TargetSize, TargetSize, ResizedColors, false);
    }
    else
    {
        ResizedColors = SrcColors;
    }

    // Convert FColor (BGRA) to uint8 array (selected channel, 1 byte per pixel)
    ResultData.SetNumUninitialized(TargetSize * TargetSize);
    uint8* DestData = ResultData.GetData();
    for (int32 i = 0; i < TargetSize * TargetSize; ++i)
    {
        const FColor& C = ResizedColors[i];
        DestData[i] = GetChannelValue(C, Channel);
    }
#else
    UE_LOG(LogTexturePacker, Error, TEXT("ChannelPackerPro requires WITH_EDITORONLY_DATA to access Source."));
    ResultData.Init(0, TargetSize * TargetSize);
#endif

    return ResultData;
}

void FChannelPackerProModule::CreateTexture(const FString& PackageName, int32 Resolution)
{
    check(IsInGameThread());

    // Initialize progress dialog with 5 steps total
    FScopedSlowTask SlowTask(5.0f, LOCTEXT("ProgressProcessing", "Processing Textures..."));
    SlowTask.MakeDialog(true); // true = cancellable

    // Create the package
    UPackage* Package = CreatePackage(*PackageName);
    Package->FullyLoad();

    SlowTask.EnterProgressFrame(1.0f, LOCTEXT("ProgressPackageCreated", "Package created. Loading input textures..."));

    if (SlowTask.ShouldCancel())
    {
        FText CancelMsg = LOCTEXT("OperationCancelled", "Texture generation was cancelled by user.");
        ShowNotification(CancelMsg, false);
        return;
    }

    // Create the Texture2D
    FName TextureName = FName(*FPaths::GetBaseFilename(PackageName));
    UTexture2D* NewTexture = NewObject<UTexture2D>(Package, TextureName, RF_Public | RF_Standalone | RF_MarkAsRootSet);

    // Get Resized Data for Inputs
    SlowTask.EnterProgressFrame(1.0f, LOCTEXT("ProgressLoadingChannels", "Processing Red, Green, Blue channels..."));

    if (SlowTask.ShouldCancel())
    {
        FText CancelMsg = LOCTEXT("OperationCancelled", "Texture generation was cancelled by user.");
        ShowNotification(CancelMsg, false);
        return;
    }

    const EChannelSelection SelectedR = GetChannelFromSelection(ChannelSelectionR);
    const EChannelSelection SelectedG = GetChannelFromSelection(ChannelSelectionG);
    const EChannelSelection SelectedB = GetChannelFromSelection(ChannelSelectionB);

    TArray<uint8> DataR = GetResizedTextureData(InputTextureR.Get(), Resolution, SelectedR);
    TArray<uint8> DataG = GetResizedTextureData(InputTextureG.Get(), Resolution, SelectedG);
    TArray<uint8> DataB = GetResizedTextureData(InputTextureB.Get(), Resolution, SelectedB);

    SlowTask.EnterProgressFrame(1.0f, LOCTEXT("ProgressLoadingAlpha", "Processing Alpha channel..."));

    if (SlowTask.ShouldCancel())
    {
        FText CancelMsg = LOCTEXT("OperationCancelled", "Texture generation was cancelled by user.");
        ShowNotification(CancelMsg, false);
        return;
    }

    const EChannelSelection SelectedA = GetChannelFromSelection(ChannelSelectionA);
    TArray<uint8> DataA = GetResizedTextureData(InputTextureA.Get(), Resolution, SelectedA);

    bool bHasAlpha = InputTextureA.IsValid();

#if WITH_EDITORONLY_DATA
    // Initialize Source
    NewTexture->Source.Init(Resolution, Resolution, 1, 1, TSF_BGRA8);

    SlowTask.EnterProgressFrame(1.0f, LOCTEXT("ProgressWritingPixels", "Writing pixel data..."));

    if (SlowTask.ShouldCancel())
    {
        FText CancelMsg = LOCTEXT("OperationCancelled", "Texture generation was cancelled by user.");
        ShowNotification(CancelMsg, false);
        return;
    }

    // Lock and Write Pixels directly to Source
    uint8* MipData = NewTexture->Source.LockMip(0);
    if (MipData)
    {
        for (int32 i = 0; i < Resolution * Resolution; ++i)
        {
            // Each Data array is now Grayscale (1 byte per pixel).
            // New.R = InputR_Data[i]
            // New.G = InputG_Data[i]
            // New.B = InputB_Data[i]

            uint8 R_Val = DataR[i];
            uint8 G_Val = DataG[i];
            uint8 B_Val = DataB[i];
            uint8 A_Val = bHasAlpha ? DataA[i] : 255; // Use Red channel of Alpha input, or 255

            // Output Texture is PF_B8G8R8A8 (BGRA memory layout)
            MipData[i * 4 + 0] = B_Val; // B
            MipData[i * 4 + 1] = G_Val; // G
            MipData[i * 4 + 2] = R_Val; // R
            MipData[i * 4 + 3] = A_Val; // A
        }
    }
    NewTexture->Source.UnlockMip(0);
#endif

    SlowTask.EnterProgressFrame(1.0f, LOCTEXT("ProgressFinalizing", "Finalizing texture..."));

    if (SlowTask.ShouldCancel())
    {
        FText CancelMsg = LOCTEXT("OperationCancelled", "Texture packing cancelled by user.");
        ShowNotification(CancelMsg, false);
        return;
    }

    // Final settings
    const TextureCompressionSettings Compression = GetSelectedCompressionSettings();
    NewTexture->CompressionSettings = Compression;

    // Enable sRGB only for Default compression; keep packed masks linear otherwise.
    NewTexture->SRGB = (Compression == TC_Default);
    
    NewTexture->UpdateResource();
    NewTexture->PostEditChange();

    Package->MarkPackageDirty();
    FAssetRegistryModule::AssetCreated(NewTexture);

    FText FormatPattern = LOCTEXT("SuccessTextureSaved", "Texture Saved: {0}");
    ShowNotification(FText::Format(FormatPattern, FText::FromString(PackageName)), true);
}

void FChannelPackerProModule::ShowNotification(const FText& Message, bool bSuccess)
{
    FNotificationInfo Info(Message);
    Info.ExpireDuration = 3.0f;

    if (bSuccess)
    {
        Info.Image = FAppStyle::GetBrush("Icons.SuccessWithColor");
    }
    else
    {
        Info.Image = FAppStyle::GetBrush("Icons.ErrorWithColor");
    }

    TSharedPtr<SNotificationItem> NotificationItem = FSlateNotificationManager::Get().AddNotification(Info);
    if (NotificationItem.IsValid())
    {
        NotificationItem->SetCompletionState(bSuccess ? SNotificationItem::CS_Success : SNotificationItem::CS_Fail);
        NotificationItem->ExpireAndFadeout();
    }
}

TextureCompressionSettings FChannelPackerProModule::GetSelectedCompressionSettings() const
{
    if (CurrentCompressionOption.IsValid())
    {
        const FString& Option = *CurrentCompressionOption;
        if (Option == "Masks") return TC_Masks;
        // if (Option == "Grayscale") return TC_Grayscale;
        if (Option == "Default") return TC_Default;
    }
    return TC_Masks; // Fallback
}

#undef LOCTEXT_NAMESPACE

IMPLEMENT_MODULE(FChannelPackerProModule, ChannelPackerPro)
